<?php
/* --------------------------------------------------------------
 ModuleCenterConfigurationRepository.php 2020-01-24
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 24 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Configuration\Compatibility\Repositories;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\DBALException;
use Gambio\Core\Configuration\Compatibility\ModulesConfigurationRepository;
use Gambio\Core\Configuration\Models\Read\ConfigurationGroupItem;
use Gambio\Core\Configuration\Repositories\Components\ConfigurationMapper;

/**
 * Class ModuleCenterConfigurationRepository
 * @package Gambio\Core\Configuration\Compatibility\Repositories
 *
 * @codeCoverageIgnore
 */
class ModuleCenterConfigurationRepository implements ModulesConfigurationRepository
{
    private const KEY_PREFIX = 'configuration/';
    private const TABLE_NAME = 'gx_configurations';
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var ConfigurationMapper
     */
    private $mapper;
    
    
    /**
     * ModuleCenterConfigurationRepository constructor.
     *
     * @param Connection          $connection
     * @param ConfigurationMapper $mapper
     */
    public function __construct(Connection $connection, ConfigurationMapper $mapper)
    {
        $this->connection = $connection;
        $this->mapper     = $mapper;
    }
    
    
    /**
     * @inheritDoc
     */
    public function findByKey(string $configurationKey, string $module, string $customPrefix = null): ConfigurationGroupItem
    {
        $configurationKey = $this->prefixKey($configurationKey, $customPrefix);
        $qb               = $this->connection->createQueryBuilder();
        
        $selects       = [
            $this->connection->quoteIdentifier('key'),
            $this->connection->quoteIdentifier('value'),
            $this->connection->quoteIdentifier('type'),
            $this->connection->quoteIdentifier('l.code'),
        ];
        $select        = implode(', ', $selects);
        $joinCondition = "{$this->connection->quoteIdentifier('c.language_id')} = {$this->connection->quoteIdentifier('l.languages_id')}";
        $where         = "{$this->connection->quoteIdentifier('key')} = {$qb->createNamedParameter($configurationKey)}";
        
        $dataSet = $qb->select($select)
            ->from(self::TABLE_NAME, 'c')
            ->leftJoin('c', 'languages', 'l', $joinCondition)
            ->where($where)
            ->execute()
            ->fetch();
        
        return $this->mapper->mapItem($dataSet, $module);
    }
    
    
    /**
     * @inheritDoc
     * @throws DBALException
     */
    public function updateConfiguration(string $configurationKey, string $value, string $customPrefix = null): void
    {
        $configurationKey = $this->prefixKey($configurationKey, $customPrefix);
        $qb               = $this->connection->createQueryBuilder();
        
        $keyName   = $this->connection->quoteIdentifier('key');
        $valueName = $this->connection->quoteIdentifier('value');
        
        $where  = "{$keyName} = {$qb->createNamedParameter($configurationKey)}";
        $result = $qb->select('*')->from(self::TABLE_NAME)->where($where)->execute();
        
        if ($result->rowCount() > 0) {
            $this->connection->update(self::TABLE_NAME, [$valueName => $value], [$keyName => $configurationKey]);
            
            return;
        }
        $this->connection->insert('gx_configurations',
                                  [
                                      $keyName   => $configurationKey,
                                      $valueName => $value,
                                  ]);
    }
    
    
    private function prefixKey(string $configurationKey, ?string $customPrefix): string
    {
        $prefix = $customPrefix ? : self::KEY_PREFIX;
        
        return strpos($configurationKey, $prefix) !== 0 ? $prefix . $configurationKey : $configurationKey;
    }
}